import { Inject, Injectable } from '@nestjs/common';
import { PrimeLogger, TxType } from 'src/framework';
import { KeywordRepository, KeywordService } from 'src/licitaapp/application';
import { MercadoPublicoGirosRepository } from 'src/licitaapp/application/mercado-publico/mp-giros/mercado-publico-giros.repository';
import {
  InsertKeyword,
  KeywordItem,
  KeywordOrigin,
  Metadata,
} from 'src/licitaapp/domain';

@Injectable()
export class KeywordServiceImpl implements KeywordService {
  private readonly LOGGER = new PrimeLogger(KeywordServiceImpl.name);
  constructor(
    @Inject('KeywordRepository')
    private readonly keywordRepository: KeywordRepository,
    private readonly mpGiros: MercadoPublicoGirosRepository,
  ) {}

  updateMetadata(keywordId: number, metadata: Metadata) {
    this.LOGGER.log(
      `updateMetadata keywordId: ${keywordId} metadata: ${JSON.stringify(metadata)}`,
    );
    return this.keywordRepository.updateMetadata(keywordId, metadata);
  }
  findByCompanyId(companyId: number, isNull: boolean, tx?: TxType) {
    this.LOGGER.log(
      `findByCompanyId keyword: ${companyId} isNull: ${isNull}`,
    );
    return this.keywordRepository.findByCompanyId(companyId, isNull, tx);
  }

  getAllWithouthMetadata(tx?: TxType) {
    this.LOGGER.log(`getAllWithouthMetadata`);
    return this.keywordRepository.getAllWithouthMetadata(tx);
  }

  save(keyword: InsertKeyword, userId: number, tx?: TxType) {
    this.LOGGER.log(`save keyword: ${JSON.stringify(keyword)}`);
    return this.keywordRepository.save(keyword, userId, tx);
  }

  saveAll(keywords: InsertKeyword[], userId: number, tx?: TxType) {
    this.LOGGER.log(`saveAll keywords: ${JSON.stringify(keywords)}`);
    return this.keywordRepository.saveAll(keywords, userId, tx);
  }

  async findByName(search: string): Promise<KeywordItem[]> {
    this.LOGGER.log(`findByName keyword: ${search}`);
    if (search.length < 3) {
      return [];
    }
    return this.mpGiros
      .buscarGiros({ q: search })
      .then((response) => {
        if (!response || response.length === 0) {
          return [{ value: search, origin: KeywordOrigin.USER }];
        } else {
          return response.map((item) => ({
            value: item.text,
            origin: KeywordOrigin.MERCADO_PUBLICO,
          }));
        }
      })
      .catch(() => {
        return [{ value: search, origin: KeywordOrigin.USER }];
      });
  }
  async logicalRemove(keywordsId: number) {
    this.LOGGER.log(`logicalRemove keyword: ${keywordsId}`);
    await this.keywordRepository.logicalRemove(keywordsId);
  }

  async deleteByCompanyId(companyId: number, tx?: TxType) {
    this.LOGGER.log(`deleteByCompanyId keyword: ${companyId}`);
    await this.keywordRepository.deleteByCompanyId(companyId, tx);
  }
}
