import { forwardRef, Inject, Injectable } from '@nestjs/common';
import { PrimeLogger } from 'src/framework';
import { Subdivision } from 'src/framework/domain/entities/subdivision.entity';
import { DBConfigService } from 'src/framework/infrastructure/drizzle';
import {
  UserCompanyService,
  CompanyRepository,
  CompanyService,
  KeywordService,
} from 'src/licitaapp';
import { UserCompanyTenderService } from 'src/licitaapp/application/service/user-company-tender-service/user-company-tender-service.interface';
import { Company, CompanyFullTO, CompanyRegisterTO, Keyword } from 'src/licitaapp/domain';
import { CompanyUserTO } from 'src/licitaapp/domain/dto/company.user.to';

@Injectable()
export class CompanyServiceImpl implements CompanyService {
  private readonly LOGGER = new PrimeLogger(CompanyServiceImpl.name);
  constructor(
    @Inject('CompanyRepository')
    private readonly companyRepository: CompanyRepository,
    @Inject('UserCompanyService')
    private readonly userCompanyService: UserCompanyService,
    @Inject('KeywordService') private readonly keywordService: KeywordService,
    @Inject('UserCompanyTenderService') private readonly userCompanyTenderService: UserCompanyTenderService,
    private readonly db: DBConfigService,
  ) {}

  async deleteByUserCompany(userId: number, companyId: number): Promise<void> {
    this.LOGGER.log(
      `deleteByUserCompany userId ${userId} companyId ${companyId}`,
    );
    return await this.userCompanyService.deleteByUserCompany(userId, companyId);
  }
  async paginationByCompanyAdmin(
    page: number,
    pageSize: number,
  ): Promise<CompanyUserTO[]> {
    this.LOGGER.log(`Paginating companies by admin, page: ${page}, pageSize: ${pageSize}`);
    const companies = await this.companyRepository.paginationByCompanyAdmin(page, pageSize);
    for (const company of companies) {
      const userInfo = await this.userCompanyService.getInfoUserCompany(
        company.companyId,
      );
      company.listUsers = userInfo;
      company.amountTenders = await this.userCompanyTenderService.countActiveTendersCompany(company.companyId);
    }
    return companies;
  }
  async getCompanyIdsToRecalculateTender(): Promise<number[]> {
    this.LOGGER.log('Getting company ids to recalculate tender');
    return await this.companyRepository.getCompanyIdsToRecalculateTender();
  }
  async getSubdivisionsByCompanyId(companyId: number): Promise<Subdivision[]> {
    this.LOGGER.log(`Getting subdivisions for companyId: ${companyId}`);
    return await this.companyRepository.getSubdivisionsByCompanyId(companyId);
  }
  async updateCheckTenders(
    companyId: number,
    checkTender: boolean,
  ): Promise<boolean> {
    this.LOGGER.log(
      `updateCheckTenders companyId ${companyId} checkTender ${checkTender}`,
    );
    return await this.companyRepository.updateCheckTenders(
      companyId,
      checkTender,
    );
  }

  async getAllWithouthMetadata(): Promise<Keyword[]> {
    this.LOGGER.log('Getting keywords metadata');
    return await this.keywordService.getAllWithouthMetadata();
  }

  async updateCompany(
    companyId: number,
    userId: number,
    updateCompany: CompanyRegisterTO,
  ): Promise<CompanyFullTO | null> {
    this.LOGGER.log(`Updating company: ${updateCompany.company.id}`);

    await this.db.conn.transaction(async (tx) => {
      const companyUpdated = await this.companyRepository.update(
        companyId,
        updateCompany,
        tx,
      );

      if (!companyUpdated) {
        this.LOGGER.error('problem updating company');
        throw new Error('Problem updating company');
      }

      this.LOGGER.log('Deleted relations to subdivisions');
      await this.companyRepository.deletedCompanySubdivisions(companyId, tx);
      this.LOGGER.log('Set new relations to subdivisions');
      if ((updateCompany.listSubdivisions.length) > 0) {
        this.LOGGER.log(
          `Saving company subdivisions for company: ${companyId} `,
        );

        const newSubdivisions = updateCompany.listSubdivisions.filter(
          (subdivision) => subdivision.id !== 7777,
        );
        this.LOGGER.log(
          `Subdivisions: ${JSON.stringify(updateCompany.listSubdivisions)}`,
        );

        await this.companyRepository.saveCompanySubdivisions(
          companyId,
          newSubdivisions,
          tx,
        );
      }

      this.LOGGER.log('deleting keywords');
      await this.keywordService.deleteByCompanyId(companyId, tx);
      this.LOGGER.log(
        `Set new relations to keywords size: ${updateCompany.listKeywords.length}`,
      );
      if (updateCompany.listKeywords.length > 0) {
        await this.keywordService.saveAll(
          updateCompany.listKeywords.map((keyword) => ({
            ...keyword,
            companyId: +companyId,
            metadata: undefined,
          })),
          userId,
          tx,
        );
        this.LOGGER.log(`Saving keywords for company: ${companyId}`);
      }
    });

    return this.findByIdWithFetch(companyId);
  }
  async createCompany(
    userId: number,
    insertCompany: CompanyRegisterTO,
  ): Promise<CompanyFullTO | null> {
    this.LOGGER.log(`Creating company for user: ${userId}`);
    const newCompany = await this.db.conn.transaction(async (tx) => {
      var company = await this.findByDNI(insertCompany.company.dni);
      if (!company) {
        this.LOGGER.log(`Saving new company dni: ${insertCompany.company.dni}`);
        company = await this.companyRepository.save(insertCompany.company, tx);
      }

      if (!company) {
        this.LOGGER.error('Company not found');
        throw new Error('Company not found');
      }
      const companyId = company.id;
      const listSubdivisioneDB =
        await this.companyRepository.getSubdivisionsByCompanyId(company.id);
      listSubdivisioneDB.length > 0 &&
        this.LOGGER.log(`Adding more Subdivisions to company: ${company.id}`);

      if ((insertCompany?.listSubdivisions?.length ?? 0) > 0) {
        this.LOGGER.log(
          `Saving company subdivisions for company: ${company.id}`,
        );

        // Filter out subdivisions that already exist in listSubdivisioneDB
        const newSubdivisions = insertCompany.listSubdivisions.filter(
          (subdivision) =>
            !listSubdivisioneDB.some(
              (existing) => existing.id === subdivision.id,
            ) && subdivision.id !== 7777,
        );

        if (newSubdivisions.length > 0) {
          await this.companyRepository.saveCompanySubdivisions(
            company.id,
            newSubdivisions,
            tx,
          );
        }
      }

      const listKeywordsDB = await this.keywordService.findByCompanyId(
        company.id,
        false,
      );

      if ((insertCompany?.listKeywords?.length ?? 0) > 0) {
        this.LOGGER.log(`Saving keywords for company: ${company.id}`);

        // Filter out keywords that already exist in listKeywordsDB
        const newKeywords = insertCompany.listKeywords.filter(
          (keyword) =>
            !listKeywordsDB.some(
              (existing) => existing.value === keyword.value,
            ),
        );

        if (newKeywords.length > 0) {
          await this.keywordService.saveAll(
            newKeywords.map((keyword) => ({
              ...keyword,
              companyId: companyId,
              metadata: undefined,
            })),
            userId,
            tx,
          );
        }
      }

      this.LOGGER.log(
        `Saving user-company association for user: ${userId} and company: ${company.id}`,
      );
      await this.userCompanyService.saveUserCompany(userId, company.id, tx);

      this.LOGGER.log(`Company created successfully: ${company.id}`);
      return company;
    });
    return this.findByIdWithFetch(newCompany.id);
  }

  async findByDNI(dni: string): Promise<Company | null> {
    this.LOGGER.log(`Finding company by DNI: ${dni}`);
    return await this.companyRepository.findByDNI(dni);
  }

  async findById(id: number): Promise<Company | null> {
    this.LOGGER.log(`Finding company by id: ${id}`);
    return await this.companyRepository.findById(id);
  }

  async findByIdWithFetch(id: number): Promise<CompanyFullTO | null> {
    this.LOGGER.log(`Finding company with fetch by id: ${id}`);
    const company = await this.findById(id);

    if (!company) {
      this.LOGGER.warn(`Company not found by id: ${id}`);
      return null;
    }

    this.LOGGER.log(`Fetching keywords for company: ${company.id}`);
    const listKeywords = await this.keywordService.findByCompanyId(
      company.id,
      false,
    );
    this.LOGGER.log(`Fetching subdivisions for company: ${company.id}`);
    const listSubdivisions =
      await this.companyRepository.getSubdivisionsByCompanyId(company.id);

    this.LOGGER.log(`Company with fetch found successfully: ${company.id}`);
    return new CompanyFullTO(
      company,
      listKeywords.map((keyword) => {
        return {
          id: keyword.id,
          value: keyword.value,
          origin: keyword.origin,
          isSelected: true,
          metadata: keyword.metadata,
        };
      }),
      listSubdivisions,
    );
  }
}
