import { Injectable } from '@nestjs/common';
import { and, eq } from 'drizzle-orm';
import { PrimeLogger, schema } from 'src/framework';
import { DBConfigService } from 'src/framework/infrastructure/drizzle';
import { HistoryTenderRepository } from 'src/licitaapp';
import { HistoryMetadata, HistoryTender, TenderTO } from 'src/licitaapp/domain';
import { UserHistoryTenderRow } from 'src/licitaapp/domain/type/user-history-tender.types';
import { TenderUtil } from 'src/licitaapp/domain/util';


@Injectable()
export class HistoryTenderRepositoryImpl  implements HistoryTenderRepository {
  private readonly LOGGER = new PrimeLogger(HistoryTenderRepositoryImpl.name);
  constructor(private readonly db: DBConfigService) {}
  async erraseUserHistoryTender(tenderInfo: TenderTO): Promise<void> {
    this.LOGGER.log(`ErraseUserHistoryTender - tenderId: ${tenderInfo.tenderId}`);

    const closeTender = TenderUtil.calculateTimeFromDateString(tenderInfo.details.Fechas.FechaCierre);
    const dateInfo = closeTender.length>0 ? `Fecha de cierre: ${tenderInfo.details.Fechas.FechaCierre?.substring(0, 16).replace('T', ' ')}` : 'Fecha de cierre no informada';
    const publishDate = TenderUtil.calculateTimeFromDateString(tenderInfo.details.Fechas.FechaPublicacion);
    const datePublishInfo = publishDate.length>0 ? `\n Fecha de publicación: ${tenderInfo.details.Fechas.FechaPublicacion?.substring(0, 16).replace('T', ' ')}` : 'Fecha de publicación no informada';

    const historyMetadata: HistoryMetadata = {
      labelAmount: TenderUtil.convertToMillions(tenderInfo.details.MontoEstimado || 0),
      typeOfMoney : TenderUtil.convertToMillions(tenderInfo.details.MontoEstimado || 0) ==='0' ? `${tenderInfo.details.Moneda} - ${tenderInfo.details.Tipo}`: tenderInfo.details.Moneda,
      labelLastUpdated: '-',
      labelCloseTender :  dateInfo + datePublishInfo,
      description: tenderInfo.description,
      name: `${tenderInfo.code} - ${tenderInfo.details.Nombre.toUpperCase() || ''}`
    };

    await this.db.conn.update(schema.userHistoryTenderTable)
      .set({ tenderId: null, codeTender: tenderInfo.code, metadata: historyMetadata })
      .where(eq(schema.userHistoryTenderTable.tenderId, tenderInfo.tenderId)).execute();
  }
  async save(userId: number, tenderId: number): Promise<boolean> {
    this.LOGGER.log(`Save - userId: ${userId}, tenderId: ${tenderId}`);
    const existHistoryTender = await this.findByUserTender(userId, tenderId);
    
    if(existHistoryTender){
      const validatedUserHistoryTender = schema.userHistoryTenderTableUpdateSchema.parse({
        updatedAt: TenderUtil.getCurrentSystemDate(),
      });
      await this.db.conn
        .update(schema.userHistoryTenderTable)
        .set(validatedUserHistoryTender)
        .where(and(eq(schema.userHistoryTenderTable.userId, userId), 
          eq(schema.userHistoryTenderTable.tenderId, tenderId)))
    }else {
      const validatedUserHistoryTender = schema.userHistoryTenderTableInsertSchema.parse({
        userId: +userId,
        tenderId: +tenderId,
      });
      try {
        await this.db.conn
          .insert(schema.userHistoryTenderTable)
          .values(validatedUserHistoryTender);
        return Promise.resolve(true);
      } catch (error) {
        throw new Error(error);
      }
    }
    return Promise.resolve(false);
  }

  async findByUserTender(userId: number, tenderId: number): Promise<UserHistoryTenderRow | undefined> {
    this.LOGGER.log(`findByUserTender - userId: ${userId}, tenderId: ${tenderId}`);
    return await this.selectUserHistoryTenderQuery()
    .where(and(eq(schema.userHistoryTenderTable.userId, userId),
      eq(schema.userHistoryTenderTable.active, true), 
      eq(schema.userHistoryTenderTable.tenderId, tenderId)))
      .then((rows: UserHistoryTenderRow[]) => {
        if (rows.length === 0) {
          return undefined;
        }
        return rows[0];
      });
  }

  private selectUserHistoryTenderQuery() {
    return this.db.conn
      .select({
        userId: schema.userHistoryTenderTable.userId,
        tenderId: schema.userHistoryTenderTable.tenderId,
        updatedAt: schema.userHistoryTenderTable.updatedAt,
        createdAt: schema.userHistoryTenderTable.createdAt,
        codeTender: schema.userHistoryTenderTable.codeTender,
      })
      .from(schema.userHistoryTenderTable)
      .$dynamic();
  }

  async getListTenderHistory(userId: number): Promise<HistoryTender[]> {
    this.LOGGER.log(`GetListTenderHistory favorites only - userId: ${userId}`);
    return await this.db.conn
      .select({
        userId: schema.userCompanyTenderTable.userId,
        tenderId: schema.userCompanyTenderTable.tenderId,
        code: schema.tenderTable.code,
      })
      .from(schema.userCompanyTenderTable)
      .innerJoin(
        schema.tenderTable,
        eq(schema.userCompanyTenderTable.tenderId, schema.tenderTable.id),
      )
      .where(and(
        eq(schema.userCompanyTenderTable.userId, userId), 
        eq(schema.userCompanyTenderTable.active, true),
        eq(schema.userCompanyTenderTable.isFavorite, true))
      )
      .then((rows) => {
        if (rows.length === 0) {
          return [];
        }
        return rows.map((row) => {
          return new HistoryTender(row.userId, row.tenderId, row.code);
        });
      });
  }

}
