import { forwardRef, Inject, Injectable } from '@nestjs/common';
import { DBConfigService, PrimeLogger, TxType } from 'src/framework';
import { ApplicationLogRepository } from 'src/licitaapp/application/repository/application-log-repository/application-log-repository.interface';
import { InsertApplicationLog, ApplicationLog } from 'src/licitaapp/domain';
import { ApplicationTypeEnum } from 'src/licitaapp/domain/enum/enum.definition';
import { schema } from 'src/framework';
import { TenderUtil } from 'src/licitaapp/domain/util';
import { and, eq } from 'drizzle-orm';

@Injectable()
export class ApplicationLogRepositoryImpl implements ApplicationLogRepository {
    private readonly LOGGER = new PrimeLogger(ApplicationLogRepositoryImpl.name);
    constructor(@Inject(forwardRef(() => DBConfigService)) private readonly db: DBConfigService) {}
    async save(applicationLog: InsertApplicationLog, tx?: TxType): Promise<number> {
        this.LOGGER.log(`save repository: ${applicationLog}`);
        const validatedApplicationLog = schema.applicationLogTableInsertSchema.parse({
            userName: applicationLog.userName,
            statusTypeId: applicationLog.statusTypeId,
            detail: applicationLog.detail,
            type: applicationLog.type
        });
        const insertedApplicationLogId = await (tx || this.db.conn)
            .insert(schema.applicationLogTable)
            .values(validatedApplicationLog)
            .$returningId()
            .then((rows) => {
                return rows[0].id;
            });

        return insertedApplicationLogId;
    }
    async updateState(applicationLogId: number, statusTypeId: number, detail: string, tx?: TxType): Promise<boolean> {
        this.LOGGER.log(`update applicationLog Id: ${applicationLogId} statusTypeId: ${statusTypeId}`);
        const validatedApplicationLog = schema.applicationLogTableUpdateSchema.parse({
            statusTypeId: statusTypeId,
            updatedAt: TenderUtil.getCurrentSystemDate(),
            detail: detail
        });
        await (tx || this.db.conn)
            .update(schema.applicationLogTable)
            .set(validatedApplicationLog)
            .where(eq(schema.applicationLogTable.id, applicationLogId))
            .execute();
        return true;
    }
    async logicalRemove(applicationLogId: number, tx?: TxType): Promise<boolean> {
        this.LOGGER.log(`logicalRemove applicationLog Id: ${applicationLogId}`);
        await (tx || this.db.conn)
            .update(schema.applicationLogTable)
            .set({ active: false, deletedAt: TenderUtil.getCurrentSystemDate() })
            .where(eq(schema.applicationLogTable.id, applicationLogId))
            .execute();
        return true;
    }
    async findAll(page: number, pageSize: number, type?: ApplicationTypeEnum): Promise<ApplicationLog[]> {
        const offset = (page - 1) * pageSize;
        this.LOGGER.log(`findAll applicationLog offset: ${offset} page: ${page}, pageSize: ${pageSize}, type: ${type}`);
        const selectFields = {
            id: schema.applicationLogTable.id,
            userName: schema.applicationLogTable.userName,
            createdAt: schema.applicationLogTable.createdAt,
            updatedAt: schema.applicationLogTable.updatedAt,
            type: schema.applicationLogTable.type,
            detail: schema.applicationLogTable.detail,
            statusText: schema.typeMasterTable.name,
        };
        
        const whereSentence = type
            ? and(eq(schema.applicationLogTable.active, true), eq(schema.applicationLogTable.type, type))
            : eq(schema.applicationLogTable.active, true);
            
        const query = this.db.conn
        .select(selectFields)
        .from(schema.applicationLogTable)
        .innerJoin(
            schema.typeMasterTable,
            eq(schema.typeMasterTable.id, schema.applicationLogTable.statusTypeId),
        )
        .where(whereSentence)
        .limit(Number(pageSize))
        .offset(offset);
        
        const rows = await query;

        if (rows.length === 0) {
        return [];
        }

        return rows.map(
        (row) =>{
            const appLog = new ApplicationLog(
            row.id,
            row.userName,
            0, 
            row.detail,
            row.type);
            appLog.createdAt = row.createdAt;
            appLog.updatedAt = row.updatedAt;
            appLog.statusText = row.statusText;
            return appLog;
        }
        );
    }

}
