import { Controller, Delete, Get, Inject, Param } from '@nestjs/common';
import { ApiOkResponse, ApiParam, ApiTags } from '@nestjs/swagger';
import { PrimeLogger, User } from 'src/framework';
import {
  AllowRoles,
  RoleShortNameEnum,
} from 'src/framework/infrastructure/decorators/allow-profiles/allow-roles.decorator';
import { ApiDefaultResponses } from 'src/framework/infrastructure/decorators/api-default-responses/api-default-responses.decorator';
import { PrimeUser } from 'src/framework/infrastructure/decorators/prime-user/prime-user.decorator';
import { CompanyService, KeywordService } from 'src/licitaapp';
import { KeywordItem } from 'src/licitaapp/domain';

@Controller('keyword')
@ApiTags('keyword')
export class KeywordController {
  private readonly LOGGER = new PrimeLogger(KeywordController.name);
  constructor(
    @Inject('KeywordService') private readonly keywordService: KeywordService,
    @Inject('CompanyService') private readonly companyService: CompanyService,
  ) {}

  @Get(':search')
  @ApiOkResponse({ type: [KeywordItem] })
  @ApiDefaultResponses()
  @ApiParam({ name: 'search', type: 'string' })
  async findAllByName(@Param('search') search: string): Promise<KeywordItem[]> {
    return this.keywordService.findByName(search);
  }

  @Delete(':keywordId/:companyId')
  @ApiDefaultResponses()
  @ApiOkResponse({ type: String })
  @AllowRoles(RoleShortNameEnum.USER)
  async logicalRemoveKeyword(
    @PrimeUser() user: User,
    @Param('keywordId') keywordId: number,
    @Param('companyId') companyId: number,
  ): Promise<String> {
    this.LOGGER.log(
      `logicalRemove keyword: ${keywordId} by user: ${user.id} companyid: ${companyId}`,
    );
    await this.keywordService.logicalRemove(keywordId);
    await this.companyService.updateCheckTenders(companyId, true);
    return 'true';
  }
}
