import { Body, Controller, Delete, Get, Inject, Param, Put } from '@nestjs/common';
import { ApiBody, ApiOkResponse, ApiParam, ApiTags } from '@nestjs/swagger';
import { PrimeLogger, User } from 'src/framework';
import { AllowRoles, RoleShortNameEnum } from 'src/framework/infrastructure/decorators/allow-profiles/allow-roles.decorator';
import { ApiDefaultResponses } from 'src/framework/infrastructure/decorators/api-default-responses/api-default-responses.decorator';
import { PrimeUser } from 'src/framework/infrastructure/decorators/prime-user/prime-user.decorator';
import { CompanyService, UserCompanyService } from 'src/licitaapp';
import { TenderService } from 'src/licitaapp/application/service/tender-service/tender-service.interface';
import { Company, CompanyFullTO, CompanyRegisterTO } from 'src/licitaapp/domain';
import { CompanyUserTO } from 'src/licitaapp/domain/dto/company.user.to';

@ApiTags('company')
@Controller('company')
export class CompanyController {
  private readonly LOGGER = new PrimeLogger(CompanyController.name);
  constructor(
    @Inject('CompanyService') private readonly companyService: CompanyService,
    @Inject('UserCompanyService') private readonly userCompanyService: UserCompanyService,
    @Inject('TenderService') private readonly tenderService: TenderService,
  ) {}

  @Get('companies-by-user')
  @ApiDefaultResponses()
  @AllowRoles(RoleShortNameEnum.USER)
  @ApiOkResponse({ type: [Company] })
  async companiesByUserId(
    @PrimeUser() user: User,
  ): Promise<Company[]> {
    this.LOGGER.warn(`Finding if user: ${user.id} have any companies`);
    return await this.userCompanyService.findCompaniesByUserId(user.id);
  }

  @Put('check-tender/:id')
  @ApiDefaultResponses()
  @ApiOkResponse({ type: () => CompanyFullTO })
  @ApiParam({ name: 'id', type: 'number' })
  async activeCheckTender(@Param('id') id: number): Promise<String> {
    this.LOGGER.warn(`ActiveCheckTender company ${id}`);
    this.companyService.updateCheckTenders(id, true);
    return 'true';
  }

  @Put(':id')
  @ApiDefaultResponses()
  @ApiOkResponse({ type: () => CompanyFullTO })
  @ApiParam({ name: 'id', type: 'number' })
  @ApiBody({ type: CompanyRegisterTO })
  async update(
    @PrimeUser() user: User,
    @Param('id') id: number,
    @Body() companyData: CompanyRegisterTO,
  ): Promise<CompanyFullTO | null> {
    this.LOGGER.warn(`Updating company ${id}`);
    const outputObj = await this.companyService.updateCompany(id, user.id, companyData);
    this.tenderService.createDashboardCompany(id, user.id);
    return outputObj;
  }

  @Get(':id')
  @ApiDefaultResponses()
  @AllowRoles(RoleShortNameEnum.USER)
  @ApiParam({ name: 'id', type: 'number' })
  @ApiOkResponse({ type: () => CompanyFullTO })
  async findById(@Param('id') id: number): Promise<CompanyFullTO | null> {
    this.LOGGER.warn(`findById id ${id}`);
    return this.companyService.findByIdWithFetch(id);
  }

  @Get(':page/:pageSize')
  @ApiDefaultResponses()
  @AllowRoles(RoleShortNameEnum.USER)
  @ApiOkResponse({ type: [Company] })
  @ApiParam({ name: 'page', type: 'number' })
  @ApiParam({ name: 'pageSize', type: 'number' })
  async paginationByUser(@PrimeUser() user: User, @Param('page') page: number, @Param('pageSize') pageSize: number ): Promise<Company[]> {
    this.LOGGER.warn(`paginationByUser user ${user.id} page ${page} pageSize ${pageSize}`);
    return await this.userCompanyService.paginationByUser(user.id, page, pageSize);
  }

  @Get()
  @ApiDefaultResponses()
  @AllowRoles(RoleShortNameEnum.USER)
  @ApiOkResponse({ type: [Company] })
  async findByUserId(@PrimeUser() user: User): Promise<Company[]> {
    this.LOGGER.warn(`findByUserId user ${user.id}`);
    return await this.userCompanyService.findCompaniesByUserId(user.id);
  }

  @Delete('relation/:companyId/:userId')
  @ApiDefaultResponses()
  @ApiOkResponse({ type: String })
  @AllowRoles(RoleShortNameEnum.USER)
  async logicalRemove(@PrimeUser() user: User, @Param('companyId') companyId: number, 
  @Param('userId') userId: number
): Promise<String> {
    this.LOGGER.warn(`delete companyId ${companyId} user ${userId}`);
    await this.userCompanyService.deleteByUserCompany(userId==0? user.id: userId, companyId);
    return 'true';
  }

  @Get('paginated-admin/:page/:pageSize')
  @ApiDefaultResponses()
  @AllowRoles(RoleShortNameEnum.USER, RoleShortNameEnum.ADMIN)
  @ApiOkResponse({ type: [CompanyUserTO] })
  @ApiParam({ name: 'page', type: 'number' })
  @ApiParam({ name: 'pageSize', type: 'number' })
  async paginationCompanyAdmin(
    @Param('page') page: number,
    @Param('pageSize') pageSize: number,
  ): Promise<CompanyUserTO[]> {
    this.LOGGER.warn(
      `paginationByCompanyAdmin page ${page} pageSize ${pageSize}`,
    );
    return await this.companyService.paginationByCompanyAdmin(page, pageSize);
  }

}
