import { HttpModule, HttpService } from '@nestjs/axios';
import { ConfigModule, ConfigService } from '@nestjs/config';
import { Test, TestingModule } from '@nestjs/testing';
import { AxiosResponse } from 'axios';
import { Observable, of, throwError } from 'rxjs';
import { MercadoPublicoHttpConfigService } from './mp.http';
import { MercadoPublicoRepository } from './mp.repository';
import { LicitacionesResponse, ListaEmpresasResponse } from './types';

describe('MercadoPublicoRepository', () => {
  let repository: MercadoPublicoRepository;
  let module: TestingModule;
  let httpService: HttpService;

  beforeAll(async () => {
    module = await Test.createTestingModule({
      imports: [
        ConfigModule.forRoot({
          isGlobal: true,
          envFilePath: ['.env.dev'],
        }),
        HttpModule.registerAsync({
          imports: [ConfigModule],
          useClass: MercadoPublicoHttpConfigService,
          inject: [ConfigService],
        }),
      ],
      providers: [MercadoPublicoRepository],
    }).compile();

    repository = module.get<MercadoPublicoRepository>(MercadoPublicoRepository);
    httpService = module.get<HttpService>(HttpService);
  });

  afterAll(() => {
    module.close();
  });

  it('should be defined', () => {
    expect(repository).toBeDefined();
  });

  describe('list', () => {
    afterEach(() => {
      jest.restoreAllMocks();
    });

    it('should call get "licitaciones" request with param and map data types properly', async () => {
      const spy = jest.spyOn(httpService, 'get').mockImplementationOnce(() =>
        of({
          data: {
            Cantidad: 1,
            FechaCreacion: '2024-07-21T10:47:53.7576991Z',
            Listado: [
              {
                Informada: 1,
                VisibilidadMonto: 1,
                SubContratacion: 1,
                ExtensionPlazo: 1,
                EsBaseTipo: 1,
                EsRenovable: 1,
                FechaCierre: '2014-01-27T15:54:00',
              },
            ],
          },
        } as unknown as AxiosResponse<LicitacionesResponse>),
      );
      const response = await repository.licitaciones({
        fecha: '02022014',
      });
      expect(spy).toHaveBeenCalledWith('/licitaciones.json', {
        params: {
          fecha: '02022014',
        },
      });
      expect(response).toBeDefined();
      expect(response?.Listado[0].Informada).toBe(true);
      expect(response?.Listado[0].VisibilidadMonto).toBe(true);
      expect(response?.Listado[0].SubContratacion).toBe(true);
      expect(response?.Listado[0].ExtensionPlazo).toBe(true);
      expect(response?.Listado[0].EsBaseTipo).toBe(true);
      expect(response?.Listado[0].EsRenovable).toBe(true);
      expect(response?.FechaCreacion).toBeInstanceOf(Date);
    });

    it('should fail if query params are more than one', async () => {
      await expect(
        repository.licitaciones({
          fecha: '02022014',
          codigo: '123',
        }),
      ).rejects.toThrow();
    });

    it('should return null if request fails', async () => {
      jest
        .spyOn(httpService, 'get')
        .mockReturnValue(
          throwError(() => new Error('Request failed')) as Observable<
            AxiosResponse<any>
          >,
        );
      await expect(
        repository.licitaciones({
          fecha: '02022014',
        }),
      ).rejects.toThrow();
    });
  });

  describe('buscarProveedor', () => {
    afterEach(() => {
      jest.restoreAllMocks();
    });

    it('should call get "BuscarProveedor" and return response', async () => {
      const spy = jest.spyOn(httpService, 'get').mockImplementationOnce(() =>
        of({
          data: {
            Cantidad: 1,
            FechaCreacion: '2024-07-21T10:47:53.7576991Z',
            listaEmpresas: [
              {
                CodigoEmpresa: '1',
                NombreEmpresa: 'Empresa 1',
              },
            ],
          },
        } as unknown as AxiosResponse<ListaEmpresasResponse>),
      );
      const response = await repository.buscarProveedor('70.017.820-k');
      expect(spy).toHaveBeenCalledWith('/empresas/BuscarProveedor', {
        params: {
          rutempresaproveedor: '70.017.820-k',
        },
      });
      expect(response).toBeDefined();
      expect(response?.listaEmpresas[0].CodigoEmpresa).toBe('1');
    });

    it('should return null if the provider does not exist', async () => {
      jest
        .spyOn(httpService, 'get')
        .mockReturnValue(
          throwError(() => new Error('Request failed')) as Observable<
            AxiosResponse<any>
          >,
        );
      const response = await repository.buscarProveedor('123456789');
      expect(response).toBeUndefined();
    });
  });

  describe('buscarComprador', () => {
    beforeEach(() => {
      jest.restoreAllMocks();
    });

    it('should call get "BuscarComprador" and return response', async () => {
      const spy = jest.spyOn(httpService, 'get').mockImplementationOnce(() =>
        of({
          data: {
            Cantidad: 1,
            FechaCreacion: '2024-07-21T10:47:53.7576991Z',
            listaEmpresas: [
              {
                CodigoEmpresa: '1',
                NombreEmpresa: 'Empresa 1',
              },
            ],
          },
        } as unknown as AxiosResponse<ListaEmpresasResponse>),
      );
      const response = await repository.buscarComprador();
      expect(spy).toHaveBeenCalledWith('/empresas/BuscarComprador');
      expect(response).toBeDefined();
      expect(response?.listaEmpresas[0].CodigoEmpresa).toBe('1');
    });

    it('should return null if request fails', async () => {
      jest
        .spyOn(httpService, 'get')
        .mockReturnValue(
          throwError(() => new Error('Request failed')) as Observable<
            AxiosResponse<any>
          >,
        );
      const response = await repository.buscarComprador();
      expect(response).toBeUndefined();
    });
  });

  describe('ordenesDeCompra', () => {
    afterEach(() => {
      jest.restoreAllMocks();
    });

    it('should call get "ordenes de compra" request with param', async () => {
      // spy and mock OrdenesDeCompraResponse response
      const spy = jest.spyOn(httpService, 'get').mockImplementationOnce(() =>
        of({
          data: {
            Cantidad: 1,
            FechaCreacion: '2024-07-21T10:47:53.7576991Z',
            Listado: [
              {
                Codigo: '1123-109-SE13',
                Fechas: { FechaCreacion: '2013-07-05T12:59:15.443' },
                TieneItems: '1',
              },
            ],
          },
        } as unknown as AxiosResponse<LicitacionesResponse>),
      );
      await repository.ordenesDeCompra({
        codigo: '1123-109-SE13',
      });
      expect(spy).toHaveBeenCalledWith('/ordenesdecompra.json', {
        params: {
          codigo: '1123-109-SE13',
        },
      });
    });

    it('should return null if request fails', async () => {
      jest
        .spyOn(httpService, 'get')
        .mockReturnValue(
          throwError(() => new Error('Request failed')) as Observable<
            AxiosResponse<any>
          >,
        );
      await expect(
        repository.ordenesDeCompra({
          fecha: '123',
        }),
      ).rejects.toThrow();
    });
  });
});
