import { HttpModule, HttpService } from '@nestjs/axios';
import { ConfigModule, ConfigService } from '@nestjs/config';
import { Test, TestingModule } from '@nestjs/testing';
import { AxiosResponse } from 'axios';
import { Observable, of, throwError } from 'rxjs';
import { MercadoPublicoGirosRepository } from './mercado-publico-giros.repository';
import { MercadoPublicoGirosHttpConfigService } from './mp-giros.http';
import { Rubro } from './types';

describe('MercadoPublicoGirosRepository', () => {
  let repository: MercadoPublicoGirosRepository;
  let httpService: HttpService;

  beforeEach(async () => {
    const module: TestingModule = await Test.createTestingModule({
      imports: [
        ConfigModule.forRoot({
          isGlobal: true,
          envFilePath: ['.env.dev'],
        }),
        HttpModule.registerAsync({
          imports: [ConfigModule],
          useClass: MercadoPublicoGirosHttpConfigService,
          inject: [ConfigService],
        }),
      ],
      providers: [MercadoPublicoGirosRepository],
    }).compile();

    repository = module.get<MercadoPublicoGirosRepository>(
      MercadoPublicoGirosRepository,
    );
    httpService = module.get<HttpService>(HttpService);
  });

  it('should be defined', () => {
    expect(repository).toBeDefined();
  });

  describe('buscarGiros', () => {
    afterEach(() => {
      jest.restoreAllMocks();
    });

    it('should call get "BuscarRubros" request with param and return data', async () => {
      const spy = jest.spyOn(httpService, 'get').mockImplementationOnce(() =>
        of({
          data: [
            {
              id: '1',
              text: 'test',
            },
          ],
        } as unknown as AxiosResponse<Rubro[]>),
      );
      const response = await repository.buscarGiros({ q: 'test' });

      expect(spy).toHaveBeenCalledWith('/BuscarRubros', {
        params: { q: 'test' },
      });
      expect(response).toBeDefined();
    });

    it('should throw error when request fails', async () => {
      const spy = jest
        .spyOn(httpService, 'get')
        .mockReturnValue(
          throwError(() => new Error('Request failed')) as Observable<
            AxiosResponse<any>
          >,
        );

      await expect(repository.buscarGiros({ q: 'test' })).rejects.toThrow();
      expect(spy).toHaveBeenCalledWith('/BuscarRubros', {
        params: { q: 'test' },
      });
    });
  });
});
