import { Inject, Injectable } from '@nestjs/common';
import { RoleService, UserRoleRepository, UserRoleService } from 'src/framework/application';
import { Role, UserPage, UserRoleTO } from 'src/framework/domain';
import { RoleShortNameEnum } from '../../decorators/allow-profiles/allow-roles.decorator';
import { PrimeLogger } from '../../definition';

@Injectable()
export class UserRoleImplService implements UserRoleService {
  private readonly LOGGER = new PrimeLogger(UserRoleImplService.name);
  constructor(
    @Inject('UserRoleRepository') private readonly userRoleRepository: UserRoleRepository,
    @Inject('RoleService') private readonly roleService: RoleService,
  ) {}
  getUserRoles(userId: number): Promise<Role[]> {
    this.LOGGER.log(`Fetching roles for userId: ${userId}`);
    return this.userRoleRepository.getUserRoles(userId);
  }
  async getAdminUserIds(): Promise<number[]> {
    this.LOGGER.log('Fetching admin user IDs');
    return await this.userRoleRepository.getAdminUserIds();
  }
  async getAllPaginated(page: number, pageSize: number, profile?: RoleShortNameEnum): Promise<UserPage[]> {
    this.LOGGER.log(`Fetching paginated user roles for page: ${page}, pageSize: ${pageSize}, profile: ${profile}`);
    const profileObj = await this.roleService.getRoleByKey(profile!);
    if (!profileObj) {
      throw new Error(`Profile with short name ${profile} not found`);
    }
    return await this.userRoleRepository.getAllPaginated(page, pageSize, profileObj.id);
  }
  async updateProfiles(params: UserRoleTO, profileId: number): Promise<void> {
    this.LOGGER.log(`Updating user roles for userId: ${params.userId}, profileId: ${profileId}, isAdded: ${params.isAdded}`);
    await this.userRoleRepository.updateRoles(params, profileId);
  }
  async getRolesByUserId(userId: number): Promise<string[]> {
    this.LOGGER.log(`Fetching roles for userId: ${userId}`);
    return await this.userRoleRepository.getRolesByUserId(userId);
  }
  async createUserRole(userId: number, roleId: number): Promise<boolean> {
    this.LOGGER.log(`Creating user role with userId: ${userId}, roleId: ${roleId}`);
    return await this.userRoleRepository.save(userId, roleId);
  }
}
