import { Injectable } from '@nestjs/common';
import { EmailService } from 'src/framework/application';
import { EmailTemplate } from 'src/framework/domain';
import { PrimeLogger } from '../../definition/logger/app.exception.logger';
import { ConfigService } from '@nestjs/config';
import { MailerService } from '@nestjs-modules/mailer';
import * as fs from 'fs/promises';
import { inspect } from 'util';

@Injectable()
export class EmailServiceImpl implements EmailService {
  private readonly LOGGER = new PrimeLogger(EmailServiceImpl.name);
  constructor(
    private readonly mailerService: MailerService,
    private readonly configService: ConfigService,
  ) {}
  async sendEmail({ to, subject }: EmailTemplate): Promise<void> {
    try {
      this.LOGGER.log('Sending email to ' + to);
      const pathBase = this.configService.get<string>('BASE_TEMPLATE_PATH');
      this.LOGGER.log('Path base templates: ' + pathBase);
      let template = await fs.readFile(`${pathBase}/BaseTemplate.html`, 'utf8');
      await this.mailerService.sendMail({
        to: to,
        subject: subject,
        html: template,
      });
    } catch (error) {
      this.LOGGER.error(`Failed to send email to ${to}: ${inspect(error)}`);
    }
  }
  async sendNewsEmail(
    to: string,
    subject: string,
    dayOfweekLabel: string,
    amountWordsTender: number = 0,
    amountGeoTenders: number = 0,
  ): Promise<void> {
    try {
      this.LOGGER.log('Sending email to ' + to);
      const pathBase = this.configService.get<string>('BASE_TEMPLATE_PATH');
      this.LOGGER.log(
        'Path base templates: ' +
          pathBase +
          '-' +
          dayOfweekLabel +
          '-' +
          amountWordsTender +
          '-' +
          amountGeoTenders,
      );
      let template = await fs.readFile(`${pathBase}/NewsTemplate.html`, 'utf8');
      await this.mailerService.sendMail({
        to: to,
        subject: subject,
        html: template
          .replace('DAY_WEEK', dayOfweekLabel)
          .replace('XXX', amountWordsTender.toString())
          .replace('YYY', amountGeoTenders.toString()),
      });
    } catch (error) {
      this.LOGGER.error(
        `Failed to send news email to ${to}: ${inspect(error)}`,
      );
    }
  }

  async sendAdminEmail(
    to: string,
    subject: string,
    message: string,
  ): Promise<void> {
    try {
      this.LOGGER.log('Sending email to ' + to);
      const pathBase = this.configService.get<string>('BASE_TEMPLATE_PATH');
      this.LOGGER.log('Path base templates: ' + pathBase + '-' + message);
      let template = await fs.readFile(
        `${pathBase}/AdminTemplate.html`,
        'utf8',
      );
      await this.mailerService.sendMail({
        to: to,
        subject: subject,
        html: template.replace('MESSAGE', message),
      });
    } catch (error) {
      this.LOGGER.error(
        `Failed to send admin email to ${to}: ${inspect(error)}`,
      );
    }
  }

  async sendUserRequest(email: string, fullname: string): Promise<void> {
    try {
      this.LOGGER.log('SendUserRequest email to ' + email);
      const pathBase = this.configService.get<string>('BASE_TEMPLATE_PATH');
      this.LOGGER.log('Path base templates: ' + pathBase + '/UserRequest');
      let template = await fs.readFile(`${pathBase}/UserRequest.html`, 'utf8');
      template = template.replace('#FULLNAME#', fullname);
      let response = await this.mailerService.sendMail({
        to: email,
        subject: 'Requerimiento de usuario',
        html: template,
      });

      this.LOGGER.log('Email sent: ', inspect(response));
    } catch (error) {
      this.LOGGER.error(
        `Failed to send user request email to ${email}: ${inspect(error)}`,
      );
    }
  }
}
