import { Injectable } from '@nestjs/common';
import { and, eq } from 'drizzle-orm';
import { UserRoleRepository } from 'src/framework/application';
import { Role, UserPage, UserRoleTO } from 'src/framework/domain';
import { DBConfigService } from 'src/framework/infrastructure/drizzle';
import * as schema from 'src/framework/infrastructure/drizzle/migrations/schema';
import { PrimeLogger } from '../../definition';
import { RoleShortNameEnum } from '../../decorators/allow-profiles/allow-roles.decorator';

@Injectable()
export class UserRolRepositoryImpl implements UserRoleRepository {
  private readonly LOGGER = new PrimeLogger(UserRolRepositoryImpl.name);
  constructor(private readonly db: DBConfigService) {}
  async getUserRoles(userId: number): Promise<Role[]> {
    this.LOGGER.log(`Finding roles for user id: ${userId}`);
    return await this.db.conn
      .select({
        id: schema.roleTable.id,
        name: schema.roleTable.name,
        shortName: schema.roleTable.shortName,
      })
      .from(schema.roleTable)
      .innerJoin(
        schema.userRoleTable,
        eq(schema.roleTable.id, schema.userRoleTable.roleId),
      )
      .where(eq(schema.userRoleTable.userId, userId))
      .then((rows) => {
        this.LOGGER.log(`Found ${rows.length} profiles for user id: ${userId}`);
        return rows.map((row) => new Role(row.id, row.name, row.shortName));
      });
  }
  async getAllPaginated(page: number, pageSize: number, profileId: number): Promise<UserPage[]> {
    const offset = (page - 1) * pageSize;
        this.LOGGER.log(`findAll user profile offset: ${offset} page: ${page}, pageSize: ${pageSize}, profileId: ${profileId}`);
        const selectFields = {
            id: schema.userTable.id,
            name: schema.userTable.name,
            lastName: schema.userTable.lastName,
            email: schema.userTable.email,
        };
        
        const whereSentence =  and(eq(schema.userTable.active, true), eq(schema.roleTable.id, profileId));
            
        const query = this.db.conn
        .select(selectFields)
        .from(schema.userRoleTable)
        .leftJoin(
            schema.userTable,
            eq(schema.userTable.id, schema.userRoleTable.userId),
        ).leftJoin(
            schema.roleTable,
            eq(schema.roleTable.id, schema.userRoleTable.roleId),
        )
        .where(whereSentence)
        .limit(Number(pageSize))
        .offset(offset);
        
        const rows = await query;

        if (rows.length === 0) {
        return [];
        }

        return rows.map(
        (row) =>{
            const record = new UserPage();
            record.id = row.id!;
            record.name = row.name!;
            record.lastName = row.lastName;
            record.email = row.email!;
            return record;
        }
        );
  }
  async getAdminUserIds(): Promise<number[]> {
    this.LOGGER.log('Getting admin user ids');
    try {
      const adminProfiles = await this.db.conn
        .select({
          userId: schema.userRoleTable.userId
        })
        .from(schema.userRoleTable)
        .innerJoin(
          schema.roleTable,
          eq(schema.userRoleTable.roleId, schema.roleTable.id),
        )
        .where(eq(schema.roleTable.shortName, RoleShortNameEnum.ADMIN)).execute();
      return adminProfiles.map((profile) => profile.userId);
    } catch (error) {
      throw new Error(error);
    }
  }
  async updateRoles(params: UserRoleTO, profileId: number): Promise<boolean | undefined> {
    this.LOGGER.log(`Updating profiles for user id: ${params}`);
      if(params.isAdded){
        this.LOGGER.log(`Adding profile ${profileId} to user ${params.userId}`);
        return await this.save(params.userId, profileId);
      }else{
        this.LOGGER.log(`Removing profile ${profileId} from user ${params.userId}`);
        await (this.db.conn)
              .delete(schema.userRoleTable)
              .where(
                and(eq(schema.userRoleTable.userId, params.userId),
                eq(schema.userRoleTable.roleId, profileId) )
              )
              .execute();
      }    
  }
  async getRolesByUserId(userId: number): Promise<string[]> {
    this.LOGGER.log(`Getting roles for user id: ${userId}`);
    const profiles = await this.db.conn
        .select({
          shortName: schema.roleTable.shortName
        })
        .from(schema.userRoleTable)
        .innerJoin(
          schema.roleTable,
          eq(schema.userRoleTable.roleId, schema.roleTable.id),
        )
        .where(eq(schema.userRoleTable.userId, userId))
        .execute();
      return profiles.map((profile) => profile.shortName);
  }

  async save(userId: number, roleId: number): Promise<boolean> {
    this.LOGGER.log(`Saving user role with userId: ${userId}, roleId: ${roleId}`);
    try {
      await this.db.conn
        .insert(schema.userRoleTable)
        .values({ userId, roleId });
      return true;
    } catch (error) {
      throw new Error(error);
    }
  }
}
