import { Injectable } from '@nestjs/common';
import { UserCampaignsRepository } from 'src/framework/application/repository/user-campaigns-repository/user-campaigns-repository.interface';
import { DBConfigService } from '../../drizzle';
import { InsertUserCampaign, UpdateUserCampaign, UserCampaign } from 'src/framework/domain';
import { PrimeLogger } from '../../definition';
import { TenderUtil } from 'src/licitaapp/domain/util';
import { and, eq } from 'drizzle-orm';
import * as schema from 'src/framework/infrastructure/drizzle/migrations/schema';

@Injectable()
export class UserCampaignsRepositoryImpl implements UserCampaignsRepository {
    private readonly LOGGER = new PrimeLogger(UserCampaignsRepositoryImpl.name);
    constructor(private readonly db: DBConfigService) {}
    getActiveCampaigns(): Promise<UserCampaign[]> {
        this.LOGGER.log('Getting active user campaigns...');
        const query = this.db.conn
            .select()
            .from(schema.userCampaignTable)
            .where(eq(schema.userCampaignTable.active, true));
        return query;
    }
    async create(data: InsertUserCampaign): Promise<UserCampaign | null | undefined> {
        this.LOGGER.log('Creating user campaign...');
        const validateUserCampaign = schema.userCampaignTableInsertSchema.parse(data);
        const userCampaignId = await (this.db.conn)
            .insert(schema.userCampaignTable)
            .values(validateUserCampaign)
            .$returningId()
            .then((rows) => {
                return rows[0].id;
            });
        return await this.findById(userCampaignId).then((object) => {
            if (object) {
                return object;
            }
            throw new Error('User campaign not found');
        });
    }
    async findById(id: number): Promise<UserCampaign | null | undefined> {
        this.LOGGER.log(`Finding user campaign by ID: ${id}`);
        const selectFields = {
            id: schema.userCampaignTable.id,
            description: schema.userCampaignTable.description,
            title: schema.userCampaignTable.title,
            dayOfWeek: schema.userCampaignTable.dayOfWeek,
            minute: schema.userCampaignTable.defMinute,
            hour: schema.userCampaignTable.defHour,
            defaultMessage: schema.userCampaignTable.defaultMessage,
        };
        const query = this.db.conn
            .select(selectFields)
            .from(schema.userCampaignTable)
            .where(eq(schema.userCampaignTable.id, id));
        const rows = await query;
        if (rows.length === 0) {
            return null;
        }
        const row = rows[0];
        const record = new UserCampaign();
        record.id = row.id;
        record.description = row.description;
        record.title = row.title;
        record.dayOfWeek = row.dayOfWeek;
        record.defMinute = row.minute;
        record.defHour = row.hour;
        record.defaultMessage = row.defaultMessage;
        return record;
    }
    async update(data: UpdateUserCampaign): Promise<string> {
        this.LOGGER.log('Updating user campaign...');
        await this.db.conn
            .update(schema.userCampaignTable)
            .set({
                title: data.title,
                description: data.description,
                dayOfWeek: data.dayOfWeek,
                defMinute: data.defMinute,
                defHour: data.defHour,
                defaultMessage: data.defaultMessage,
                updatedAt: TenderUtil.getCurrentSystemDate(),
            })
            .where(eq(schema.userCampaignTable.id, data.id!))
            .execute();
        return 'true';
    }
    async getPagination(page: number, pageSize: number, defaultMessageFilter?: string): Promise<UserCampaign[]> {
        this.LOGGER.log(`paginationTransferVehicle - page: ${page}, pageSize: ${pageSize}, defaultMessageFilter: ${defaultMessageFilter}`);
        //TODO: implementar filter
        const offset = (page - 1) * pageSize;
        const selectFields = {        
          id: schema.userCampaignTable.id,
          description: schema.userCampaignTable.description,
          title: schema.userCampaignTable.title,
          dayOfWeek: schema.userCampaignTable.dayOfWeek,
          minute: schema.userCampaignTable.defMinute,
          hour: schema.userCampaignTable.defHour,
          defaultMessage: schema.userCampaignTable.defaultMessage,
        };
        const query = this.db.conn
          .select(selectFields)
          .from(schema.userCampaignTable)
          .where(
            and(
              eq(schema.userCampaignTable.active, true)
            ),
          )
          .limit(Number(pageSize))
          .offset(offset);

        const rows = await query;

        if (rows.length === 0) {
          return [];
        }

        return rows.map(
          (row) =>{
            const record = new UserCampaign();     
            record.id = row.id;
            record.description = row.description;
            record.title = row.title;
            record.dayOfWeek = row.dayOfWeek;
            record.defMinute = row.minute;
            record.defHour = row.hour;
            record.defaultMessage = row.defaultMessage;
            return record;
          }
        );
    }
    logicalRemove(campaignId: number): Promise<void> {
        this.LOGGER.log(`Logical removing user campaign with ID: ${campaignId}`);
        return this.db.conn
            .update(schema.userCampaignTable)
            .set({ active: false , deletedAt: TenderUtil.getCurrentSystemDate() })
            .where(eq(schema.userCampaignTable.id, campaignId))
            .execute()
            .then(() => {
                this.LOGGER.log(`User campaign with ID: ${campaignId} logically removed`);
            });
    }
}
