import {
  ArgumentsHost,
  Catch,
  ExceptionFilter,
  HttpException,
  HttpStatus,
} from '@nestjs/common';
import { ApiProperty } from '@nestjs/swagger';
import { Request, Response } from 'express';
import { ZodError, ZodIssue } from 'zod';
import { ExceptionType, PrimeException } from '../';
import { PrimeLogger } from '../logger/app.exception.logger';

export class ApiErrorResponse {
  @ApiProperty({
    example: HttpStatus.INTERNAL_SERVER_ERROR,
    enum: HttpStatus,
    enumName: 'HttpStatus',
  })
  status: number;
  @ApiProperty({ example: 'Internal server error' })
  message: string;
  @ApiProperty({
    example: ExceptionType.INTERNAL,
    enum: ExceptionType,
    enumName: 'ExceptionType',
  })
  type: ExceptionType;
}

export class ApiErrorExamples {
  static readonly RECORD_EXIST = {
    status: HttpStatus.AMBIGUOUS,
    message: 'Record already exists',
    type: ExceptionType.INTERNAL,
  }
  static readonly FORBIDEN = {
    status: HttpStatus.FORBIDDEN,
    message: 'Forbidden',
    type: ExceptionType.INTERNAL,
  };

  static readonly NOT_FOUND = {
    status: HttpStatus.NOT_FOUND,
    message: 'Not found',
    type: ExceptionType.INTERNAL,
  };

  static readonly UNAUTHORIZED = {
    status: HttpStatus.UNAUTHORIZED,
    message: 'Unauthorized',
    type: ExceptionType.INTERNAL,
  };

  static readonly BAD_REQUEST = {
    status: HttpStatus.BAD_REQUEST,
    message: 'Bad request',
    type: ExceptionType.INTERNAL,
  };

  static readonly CONFLICT = {
    status: HttpStatus.CONFLICT,
    message: 'Conflict',
    type: ExceptionType.INTERNAL,
  };

  static readonly INTERNAL = {
    status: HttpStatus.INTERNAL_SERVER_ERROR,
    message: 'Internal server error',
    type: ExceptionType.INTERNAL,
  };
}

@Catch()
export class HttpExceptionFilter implements ExceptionFilter {
  private readonly LOGGER = new PrimeLogger(HttpExceptionFilter.name);
  catch(exception: unknown, host: ArgumentsHost) {
    const ctx = host.switchToHttp();
    const response = ctx.getResponse<Response>();
    const request = ctx.getRequest<Request>();

    let status = 500;
    let message = 'Internal server error';
    let type = ExceptionType.INTERNAL;

    const errorName = this.getExceptionConstructorName(exception);

    // Extract the constructor name to use as context for logging
    let exceptionName =
      exception instanceof Error ? exception.constructor.name : 'UnknownError';

    let exceptionDetails:
      | {
          [key: string]: any;
        }
      | undefined;

    if (exception instanceof HttpException) {
      status = exception.getStatus();
      message = this.extractMessage(exception.getResponse()) ?? message;
      type =
        exception instanceof PrimeException
          ? exception.type
          : ExceptionType.INTERNAL;
    } else if (this.isAxiosError(exception)) {
      exceptionDetails = this.extractAxiosErrorDetails(exception);
      message = exceptionDetails.message;
      status = exceptionDetails.status;
      exceptionName = 'AxiosError';
    } else if (errorName === 'ZodError' && this.isZodError(exception)) {
      const zodError = exception as ZodError;
      status = HttpStatus.BAD_REQUEST;
      message = `Validation error`;
      exceptionDetails = this.formatZodIssues(zodError.issues);
      exceptionName = 'ZodError';
    }

    const stackArray = this.formatStackTrace(exception);

    this.LOGGER.error(
      `Error Details: ${JSON.stringify(
        {
          errorName,
          message: message,
          url: request.url,
          method: request.method,
          status: status,
          details: exceptionDetails,
          stack: stackArray,
        },
        null,
        2,
      )}`,
      exceptionName,
    );

    response.status(status).json({ status, message, type });
  }

  private isZodError(error: unknown): error is ZodError {
    return error instanceof ZodError;
  }

  private formatZodIssues(
    issues: ZodIssue[],
  ): { [key: string]: any } | undefined {
    if (issues.length === 0) {
      return undefined;
    }

    const formattedIssues = issues.reduce(
      (acc, issue) => {
        const path = issue.path.join('.') || 'Root';
        acc[path] = acc[path] || [];
        acc[path].push({
          code: issue.code,
          message: issue.message,
          fatal: issue.fatal,
        });
        return acc;
      },
      {} as { [key: string]: any },
    );

    return formattedIssues;
  }

  private getExceptionConstructorName(exception: unknown): string | null {
    if (
      exception &&
      typeof exception === 'object' &&
      'constructor' in exception &&
      typeof (exception as any).constructor === 'function' &&
      'name' in (exception as any).constructor
    ) {
      return (exception as { constructor: { name: string } }).constructor.name;
    }
    return null;
  }

  private extractMessage(
    response: string | { message?: string | string[] | { message: string } },
  ): string | undefined {
    if (typeof response === 'string') {
      return response;
    } else if (typeof response === 'object' && response.message) {
      if (Array.isArray(response.message)) {
        return response.message.join(', ');
      } else if (typeof response.message === 'string') {
        return response.message;
      } else if (
        typeof response.message === 'object' &&
        response.message.message
      ) {
        return response.message.message;
      }
    }
    return undefined;
  }

  private isAxiosError(exception: any): boolean {
    return exception && exception.isAxiosError;
  }

  private extractAxiosErrorDetails(exception: any): {
    [key: string]: any;
  } {
    return {
      message: exception.message,
      status: exception.response?.status,
      statusText: exception.response?.statusText,
      headers: exception.response?.headers,
      data: exception.response?.data,
      requestConfig: {
        url: exception.config?.url,
        method: exception.config?.method,
        data: exception.config?.data,
      },
    };
  }

  private formatStackTrace(exception: any): string[] | undefined {
    if (typeof exception?.stack === 'string') {
      return exception.stack.split('\n').map((line: string) => line.trim());
    }
    return [exception];
  }
}
