import { Body, Controller, Get, Inject, Param, Post, Put, Query } from '@nestjs/common';
import { ApiBody, ApiOkResponse, ApiParam, ApiQuery, ApiTags } from '@nestjs/swagger';
import { PrimeLogger } from '../../definition';
import { UserRoleService, UserService } from 'src/framework/application';
import { ApiDefaultResponses } from '../../decorators/api-default-responses/api-default-responses.decorator';
import { AllowRoles, RoleShortNameEnum } from '../../decorators/allow-profiles/allow-roles.decorator';
import { PrimeUser } from '../../decorators/prime-user/prime-user.decorator';
import { InsertUser, UpdateUser, User, UserPage, UserRoleTO } from 'src/framework/domain';

@Controller('users')
@ApiTags('users')
export class UserController {
  private readonly LOGGER = new PrimeLogger(UserController.name);
  constructor(
    @Inject('UserService') private readonly userService: UserService,
    @Inject('UserRoleService') private readonly userRoleService: UserRoleService,
  ) {}

  @Put()
  @ApiDefaultResponses()
  @AllowRoles(RoleShortNameEnum.USER, RoleShortNameEnum.ADMIN)
  @ApiOkResponse({ type: User })
  @ApiBody({ type: UpdateUser })
  async update(
    @PrimeUser() user: User,
    @Body() userData: UpdateUser,
  ): Promise<User> {
    this.LOGGER.log(`Updating user with id: ${user.id}`);
    return await this.userService.updateUser(user.id, userData);
  }

  @Post()
  @ApiDefaultResponses()
  @AllowRoles(RoleShortNameEnum.USER, RoleShortNameEnum.ADMIN)
  @ApiOkResponse({ type: User })
  @ApiBody({ type: InsertUser })
  async save(@Body() user: InsertUser): Promise<User> {
    this.LOGGER.log(`Saving new user with email: ${user.email}`);
    return await this.userService.createUser(user);
  }  

  @Get(':id')
  @ApiDefaultResponses()
  @AllowRoles(RoleShortNameEnum.USER, RoleShortNameEnum.ADMIN)
  @ApiOkResponse({ type: User })
  @ApiParam({ name: 'id', type: 'number' })
  async findById(@PrimeUser() user: User, @Param('id') userId: number): Promise<User | null | undefined> {
    const userIdParam = userId != 0 ? userId : user.id;
    this.LOGGER.log(`Finding user by id: ${userIdParam}`);
    return await this.userService.getUserById(userIdParam);
  }

  @Get()
  @ApiDefaultResponses()
  @AllowRoles(RoleShortNameEnum.USER, RoleShortNameEnum.ADMIN)
  @ApiOkResponse({ type: [User] })
  async findAll(): Promise<User[]> {
    this.LOGGER.log('Finding all users');
    return await this.userService.getAllUsers();
  }
  @Put('update-aggrement')
  @ApiDefaultResponses()
  @AllowRoles(RoleShortNameEnum.USER, RoleShortNameEnum.ADMIN)
  @ApiOkResponse({ type: String })
  async updateAggrement(
    @PrimeUser() user: User
  ): Promise<string> {
    this.LOGGER.log(`UpdateAggrement user with id: ${user.id}`);
    return await this.userService.updateAggrement(user);
  }

  @Put('update-profiles')
  @ApiDefaultResponses()
  @AllowRoles(RoleShortNameEnum.USER, RoleShortNameEnum.ADMIN)
  @ApiOkResponse({ type: String })
  async updateProfiles(
    @Body() params: UserRoleTO
  ): Promise<string> {
    this.LOGGER.log(`updateProfiles user with id: ${params}`);
    return await this.userService.updateProfiles(params);
  }

  @Get(':page/:pageSize')
  @ApiDefaultResponses()
  @AllowRoles(RoleShortNameEnum.USER, RoleShortNameEnum.ADMIN)
  @ApiOkResponse({ type: [UserPage] })
  @ApiParam({ name: 'page', type: 'number' })
  @ApiParam({ name: 'pageSize', type: 'number' })
  @ApiQuery({ name: 'profile', enum: RoleShortNameEnum, required: false })
  async findAllPaginated(
    @Param('page') page: number,
    @Param('pageSize') pageSize: number,
    @Query('profile') profile?: RoleShortNameEnum,
  ): Promise<UserPage[]> {
    this.LOGGER.log(`Finding all user profiles with page: ${page}, pageSize: ${pageSize}, profile: ${profile}`);
    return await this.userRoleService.getAllPaginated(
      page,
      pageSize,
      profile);
  }
}

