import { Body, Controller, Get, Inject, Param, Post, Put, Query } from '@nestjs/common';
import { ApiBody, ApiOkResponse, ApiParam, ApiQuery, ApiTags } from '@nestjs/swagger';
import { PrimeLogger } from '../../definition';
import { UserRequestService } from 'src/framework/application';
import { ApiDefaultResponses } from '../../decorators/api-default-responses/api-default-responses.decorator';
import { InsertUserRequest, ProgramUserRequestTO, RequestTenderTO, User, UserRequest } from 'src/framework/domain';
import { PrimeUser } from '../../decorators/prime-user/prime-user.decorator';
import { inspect } from 'util';

@Controller('user-request')
@ApiTags('user-request')
export class UserRequestController {

  private readonly LOGGER = new PrimeLogger(UserRequestController.name);
  constructor(@Inject('UserRequestService') private readonly userRequestService: UserRequestService) {}

  @Post()
  @ApiDefaultResponses()
  @ApiOkResponse({ type: UserRequest })
  @ApiBody({ type: InsertUserRequest })
  async createUserRequest(
    @PrimeUser() user: User,
    @Body() dataUserRequest: InsertUserRequest,
  ): Promise<UserRequest | null> {
    this.LOGGER.log(`Creating user request for user: ${user.id}`);
    return await this.userRequestService.createUserRequest(user.id, dataUserRequest);
  }

  @Post('program')
  @ApiDefaultResponses()
  @ApiOkResponse({ type: String })
  @ApiBody({ type: ProgramUserRequestTO })
  async createProgramUserRequest(
    @PrimeUser() user: User,
    @Body() dataUserRequest: ProgramUserRequestTO,
  ): Promise<String> {
    this.LOGGER.log(`Creating user request for user: ${user.id} ${inspect(ProgramUserRequestTO)}`);
    return await this.userRequestService.createProgramUserRequest(user.id, dataUserRequest);
  }

  

  @Get(':page/:pageSize/:shortStateUserRequest')
  @ApiDefaultResponses()
  @ApiOkResponse({ type: [RequestTenderTO] })
  @ApiParam({ name: 'page', type: 'number'  })
  @ApiParam({ name: 'pageSize', type: 'number' })
  @ApiParam({ name: 'shortStateUserRequest', type: 'string' })
  @ApiQuery({ name: 'filter', required: false, type: String, description: 'Filtro opcional para buscar requerimientos' })
  async getPaginatedUserRequest(@Param('page') page: number, 
  @Param('pageSize') pageSize: number,  @Query('filter') filter: string,
    @Param('shortStateUserRequest') shortStateUserRequest: string): Promise<RequestTenderTO[]> {
    this.LOGGER.log(`GetPaginatedUserRequest - page: ${page}, pageSize: ${pageSize}, filter: ${filter}, shortStateUserRequest: ${shortStateUserRequest}`);
    return await this.userRequestService.getPagination(page, pageSize, filter, shortStateUserRequest);
  }

  @Put(':id/:stateRequestType')
  @ApiDefaultResponses()
  @ApiOkResponse({ type: () => UserRequest })
  @ApiParam({ name: 'id', type: 'number' })
  @ApiParam({ name: 'stateRequestType', type: 'string' })
  async update(
    @Param('id') userRequestId: number,
    @Param('stateRequestType') stateRequestType: string,
  ): Promise<UserRequest | null> {
    this.LOGGER.log(`Updating UserStateOrLogicalRemove request: ${userRequestId} with state: ${stateRequestType}`);
    return this.userRequestService.updateUserStateOrLogicalRemove(userRequestId, stateRequestType);
  }
}
