import { Body, Controller, Delete, Get, Inject, Param, Post, Put, Query } from '@nestjs/common';
import { ApiBody, ApiConflictResponse, ApiOkResponse, ApiParam, ApiQuery, ApiTags } from '@nestjs/swagger';
import { ApiErrorExamples, ApiErrorResponse, PrimeLogger } from '../../definition';
import { ApiDefaultResponses } from '../../decorators/api-default-responses/api-default-responses.decorator';
import { AllowRoles, RoleShortNameEnum } from '../../decorators/allow-profiles/allow-roles.decorator';
import { ApiVehicleResponses } from '../../decorators';
import { UserCampaignsService } from 'src/framework/application';
import { InsertUserCampaign, UpdateUserCampaign, UserCampaign } from 'src/framework/domain';

@ApiTags('user-campaigns')
@Controller('user-campaigns')
export class UserCampaignsController {

    private readonly LOGGER = new PrimeLogger(UserCampaignsController.name);

    constructor(
        @Inject('UserCampaignsService') private readonly userCampaignsService: UserCampaignsService,        
    ) {}


    @Post()
    @ApiVehicleResponses()
    @ApiConflictResponse({ type: ApiErrorExamples })
    @AllowRoles(RoleShortNameEnum.USER)
    @ApiOkResponse({ type: UserCampaign })
    @ApiBody({ type: InsertUserCampaign })
    async save(
        @Body() data: InsertUserCampaign,
    ): Promise<UserCampaign | null | undefined> {
        this.LOGGER.log('save user campaign data: ' + JSON.stringify(data));
        return await this.userCampaignsService.create(data);
    }

    @Put()
    @ApiDefaultResponses()
    @ApiConflictResponse({ type: ApiErrorResponse })
    @ApiOkResponse({ type: String })
    async updateUserCampaign(@Body() data: UpdateUserCampaign): Promise<String> {
        this.LOGGER.log(`update user campaign data: ${JSON.stringify(data)}`);
        return await this.userCampaignsService.update(data);
    }

    @Get(':page/:pageSize')
    @ApiDefaultResponses()
    @AllowRoles(RoleShortNameEnum.USER)
    @ApiOkResponse({ type: [UserCampaignsController] })
    @ApiParam({ name: 'page', type: 'number'  })
    @ApiParam({ name: 'pageSize', type: 'number' })
    @ApiQuery({ name: 'filter', required: false, type: String, description: 'Filtro opcional para titulos' })
    async getPaginatedUserCampaign(@Param('page') page: number, 
    @Param('pageSize') pageSize: number, @Query('filter') filter: string): Promise<UserCampaign[]> {
        this.LOGGER.log(`getPaginatedUserCampaign - page: ${page}, pageSize: ${pageSize}, filter: ${filter}`);
        return await this.userCampaignsService.getPagination(page, pageSize, filter);
    }

    @Delete(':id')
    @ApiDefaultResponses()
    @ApiConflictResponse({ type: ApiErrorResponse })
    @ApiOkResponse({ type: String })
    @ApiParam({ name: 'id', type: 'number' })
    async logicalRemove(@Param('id') campaignId: number): Promise<String> {
        this.LOGGER.log(`logicalRemove user ${campaignId}`);
        await this.userCampaignsService.logicalRemove(campaignId);
        this.LOGGER.log(`logicalRemove user ${campaignId} deleted`);
        return 'true';
    }


}

