import { Body, Controller, Get, Inject, Param, Post } from '@nestjs/common';
import { ApiBody, ApiOkResponse, ApiParam, ApiTags } from '@nestjs/swagger';
import {
  PrimeLogger,
  EmailService,
  User,
  UserDeviceService,
  FirebaseService,
} from 'src/framework';
import { ApiDefaultResponses } from '../../decorators/api-default-responses/api-default-responses.decorator';
import {
  AllowRoles,
  RoleShortNameEnum,
} from '../../decorators/allow-profiles/allow-roles.decorator';
import { PrimeUser } from '../../decorators/prime-user/prime-user.decorator';
import { NotificationRecordService } from 'src/framework/application/service/notification-record-service/notification-record-service.interface';
import { InsertUserDevice } from 'src/framework/domain/entities/user-device.entity';
import { TestEmailTo } from 'src/framework/domain/dto/test-email.to';
import { TestNotificationTo } from 'src/framework/domain/dto/test-notification.to';
import { NotificationRecord } from 'src/framework/domain/entities/notification-record.entity';

@Controller('notification')
@ApiTags('notification')
export class NotificationController {
  private readonly LOGGER = new PrimeLogger(NotificationController.name);

  constructor(
    @Inject('EmailService')
    private readonly emailService: EmailService,
    @Inject('UserDeviceService')
    private readonly userDeviceService: UserDeviceService,
    @Inject('FirebaseService')
    private readonly firebaseService: FirebaseService,
    @Inject('NotificationRecordService')
    private readonly notificationRecordService: NotificationRecordService,
  ) {}


  @Get(':page/:pageSize')
  @ApiDefaultResponses()
  @AllowRoles(RoleShortNameEnum.USER, RoleShortNameEnum.ADMIN)
  @ApiOkResponse({ type: [NotificationRecord] })
  @ApiParam({ name: 'page', type: 'number' })
  @ApiParam({ name: 'pageSize', type: 'number' })
  async getNotificationRecordsPaginated(
    @PrimeUser() user: User,
    @Param('page') page: number,
    @Param('pageSize') pageSize: number,
  ): Promise<NotificationRecord[]> {
    this.LOGGER.log(
      `GetPaginatedNotificationRecords for user ${user.id} page ${page} pageSize ${pageSize}`,
    );
    return await this.notificationRecordService.getNotificationRecordsPaginated(
      user.id,
      page,
      pageSize,
    );
  }

  @Post('send-notification-admin')
  @ApiDefaultResponses()
  @AllowRoles(RoleShortNameEnum.USER, RoleShortNameEnum.ADMIN)
  async sendNotificationAdmin() {
    this.LOGGER.warn(`sendNotificationAdmin`);
    this.firebaseService.notifyToAdminUsers();
  } 

  @Post('save-device-token')
  @ApiDefaultResponses()
  @AllowRoles(RoleShortNameEnum.USER, RoleShortNameEnum.ADMIN)
  @ApiBody({ type: InsertUserDevice })
  @ApiOkResponse({ type: String })
  async saveDeviceToken(
    @PrimeUser() user: User,
    @Body() deviceInfo: InsertUserDevice,
  ) {
    this.LOGGER.log(`Saving device token for user ${user.id}`);
    await this.userDeviceService.saveUserDevice(user.id, deviceInfo);
    return 'true';
  }

  @Post('test-email')
  @ApiDefaultResponses()
  @AllowRoles(RoleShortNameEnum.USER, RoleShortNameEnum.ADMIN)
  @ApiBody({ type: TestEmailTo })
  async testEmail(@Body() emailParams: TestEmailTo) {
    this.LOGGER.log(`Sending email to ${emailParams.email}`);
    await this.emailService.sendAdminEmail(
      emailParams.email,
      emailParams.title,
      emailParams.body,
    );
  }
  @Post('test-notification')
  @ApiDefaultResponses()
  @ApiBody({ type: TestNotificationTo })
  @AllowRoles(RoleShortNameEnum.USER, RoleShortNameEnum.ADMIN)
  async testNotification(@Body() params: TestNotificationTo) {
    const data = { type_message: 'general_tender_alert' };
    const devicesList = await this.userDeviceService.findTokensByUserId(
      params.userId,
    );
    this.LOGGER.log(
      `Sending notification to user ${params.userId} with devices: ${devicesList.length}`,
    );
    const response = await this.firebaseService.sendPushNotification(
      devicesList,
      params.title,
      params.body,
      data,
    );
    this.LOGGER.warn(
      `Notification sent to user ${params.userId} with response: ${JSON.stringify(response)}`,
    );
  }

  

  
}
